<?php

if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}


function askForShippingFee(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $order_id = (int)($params['id'] ?? 0);

    if ($order_id <= 0) {
        sendMessage($chat_id, "خطا: شماره سفارش نامشخص است.", null, $platform);
        return;
    }

    $session_data['state'] = 'awaiting_shipping_fee_amount';
    $session_data['target_order_id'] = $order_id;
    updateUserField($user_id, 'session', json_encode($session_data));

    sendMessage(
        $chat_id, 
        "لطفا هزینه پست برای سفارش `{$order_id}` را به تومان وارد کنید (فقط عدد انگلیسی):".
        "\n(مثال: `35000` یا `0` برای ارسال رایگان)", 
        ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true, 'one_time_keyboard' => true], 
        $platform
    );
}


function handleShippingFeeInput(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $order_id = $session_data['target_order_id'] ?? null;
    $fee_str = trim($message['text'] ?? '');

    if (!$order_id) {
        sendMessage($chat_id, "خطا: شماره سفارش هدف مشخص نیست. عملیات لغو شد.", ['remove_keyboard' => true], $platform);
        updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
        showAdminMainMenu($chat_id, $user_id, $platform);
        return;
    }

    if (isValidNonNegativeNumber($fee_str)) {
        $fee = convertPersianToEnglishDigits($fee_str);
        
        $session_data['state'] = 'awaiting_shipping_fee_note';
        $session_data['shipping_fee_amount'] = $fee;
        updateUserField($user_id, 'session', json_encode($session_data));

        sendMessage(
            $chat_id,
            "مبلغ `" . number_format($fee) . "` تومان ثبت شد.\n\n".
            "لطفا یادداشت هزینه ارسال را وارد کنید (مثلا: پست پیشتاز + بسته‌بندی).\n\n".
            "برای رد شدن و ثبت بدون یادداشت، دکمه 'Skip' را بزنید.",
            ['keyboard' => [[['text' => 'Skip (بدون یادداشت)']], [['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true, 'one_time_keyboard' => true],
            $platform
        );
    } else {
        sendMessage($chat_id, "لطفا یک مبلغ عددی معتبر (بزرگتر یا مساوی صفر) برای هزینه پست وارد کنید.", null, $platform);
    }
}

function handleShippingFeeNoteInput(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $order_id = $session_data['target_order_id'] ?? null;
    $fee = $session_data['shipping_fee_amount'] ?? null;
    $note_raw = trim($message['text'] ?? '');
    
    $note = null;
    if (strtolower($note_raw) !== 'skip (بدون یادداشت)' && !empty($note_raw)) {
        $note = strip_tags($note_raw);
    }

    if ($order_id === null || $fee === null) {
        sendMessage($chat_id, "خطا: اطلاعات سفارش یا مبلغ در نشست یافت نشد. عملیات لغو شد.", ['remove_keyboard' => true], $platform);
        updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
        showAdminMainMenu($chat_id, $user_id, $platform);
        return;
    }

    if (setOrderShippingFee($order_id, $fee, $note)) {
        $formatted_fee_admin = number_format($fee);
        
        $is_free_shipping = (bccomp($fee, '0.00', BC_SCALE) == 0);
        
        if ($is_free_shipping) {
            sendMessage($chat_id, "✅ هزینه پست `0` تومان (رایگان) برای سفارش `{$order_id}` ثبت شد. وضعیت سفارش به 'در حال پردازش' تغییر یافت.", ['remove_keyboard' => true], $platform);
        } else {
            sendMessage($chat_id, "✅ هزینه پست به مبلغ `{$formatted_fee_admin}` تومان برای سفارش `{$order_id}` ثبت شد. وضعیت به 'منتظر هزینه پست' تغییر یافت.", ['remove_keyboard' => true], $platform);
        }

        $order_owner = getOrderOwner($order_id);
        if ($order_owner) {
            $order = getOrderDetails($order_id);
            if ($order) {
                
                $product_total_amount = $order['total_amount'] ?? '0.00';
                $shipping_fee_from_db = $order['shipping_fee'] ?? '0.00';
                $new_display_total = bcadd($product_total_amount, $shipping_fee_from_db, BC_SCALE);
                
                $formatted_display_total = number_format($new_display_total);
                $formatted_shipping_fee = number_format($shipping_fee_from_db);
                
                $user_message = "";
                $user_buttons = [];
                
                if ($is_free_shipping) {
                    $user_message = "🚚 هزینه ارسال سفارش شما (`{$order_id}`) *رایگان* محاسبه شد.\n\n";
                    if ($note) {
                        $user_message .= "یادداشت ارسال: " . htmlspecialchars($note) . "\n\n";
                    }
                    $user_message .= "سفارش شما به زودی پردازش خواهد شد.";
                } else {
                    $user_message = "🚚 هزینه پست برای سفارش شما (`{$order_id}`) مبلغ `{$formatted_shipping_fee}` تومان تعیین شد.\n\n";
                    if ($note) {
                        $user_message .= "یادداشت ارسال: " . htmlspecialchars($note) . "\n\n";
                    }
                    $user_message .= "مبلغ نهایی سفارش شما (با احتساب هزینه پست): `{$formatted_display_total}` تومان";
                    
                    $payment_page_url = PAYMENT_PAGE_URL . '?order_id=' . $order_id;
                    $user_message .= "\n\n👇 لطفا برای تکمیل سفارش، هزینه را پرداخت نمایید:";
                    
                    $user_buttons = [[['text' => '💰 پرداخت هزینه ارسال (' . $formatted_shipping_fee . ' تومان)', 'url' => $payment_page_url]]];
                }
                
                $reply_markup = !empty($user_buttons) ? ['inline_keyboard' => $user_buttons] : null;

                if (!empty($order_owner['telegram_chat_id'])) {
                    sendMessage($order_owner['telegram_chat_id'], $user_message, $reply_markup, 'telegram');
                }
                if (!empty($order_owner['bale_chat_id'])) {
                    sendMessage($order_owner['bale_chat_id'], $user_message, $reply_markup, 'bale');
                }
            } else {
                if (defined('ERROR_LOG_PATH')) file_put_contents(ERROR_LOG_PATH, date('Y-m-d H:i:s') . " - Error: Could not fetch order {$order_id} after setting shipping fee for user notification.\n", FILE_APPEND);
            }
        }

        updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'target_order_id' => null, 'shipping_fee_amount' => null]));
        showAdminMainMenu($chat_id, $user_id, $platform);

    } else {
        sendMessage($chat_id, "❌ خطا در ثبت هزینه پست در دیتابیس. لطفا دوباره تلاش کنید.", null, $platform);
    }
}
?>