<?php

// Prevent direct script access
if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}

/**
 * Asks the admin to enter the tracking code for an order.
 */
function askForTrackingCode(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $order_id = (int)($params['id'] ?? 0);

    if ($order_id <= 0) {
        sendMessage($chat_id, "خطا: شماره سفارش نامشخص است.", null, $platform);
        return;
    }

    // Set state
    $session_data['state'] = 'awaiting_tracking_code';
    $session_data['target_order_id'] = $order_id;
    updateUserField($user_id, 'session', json_encode($session_data));

    // Remove previous inline keyboard if exists
    // If called from ViewOrder, message_id should be present
    // Consider just sending a new message instead of complex edits
    // if ($message_id) {
    //     editMessage($chat_id, $message_id, "..."); // Or delete?
    // }

    sendMessage(
        $chat_id, 
        "لطفا کد رهگیری پستی برای سفارش `{$order_id}` را وارد کنید:", 
        ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
        $platform
    );
}

/**
 * Handles the tracking code input from the admin, saves it, and notifies the user.
 */
function handleTrackingCodeInput(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $order_id = $session_data['target_order_id'] ?? null;
    $tracking_code = trim($message['text'] ?? '');

    if (!$order_id) {
        sendMessage($chat_id, "خطا: شماره سفارش هدف مشخص نیست. عملیات لغو شد.", ['remove_keyboard' => true], $platform);
        updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
        showAdminMainMenu($chat_id, $user_id, $platform);
        return;
    }

    if (!empty($tracking_code)) {
        // Sanitize tracking code if needed
        $tracking_code = strip_tags($tracking_code);

        if (setOrderTrackingCode($order_id, $tracking_code)) { // Assumes OrderQueries loaded (also sets status to 'shipped')
             sendMessage($chat_id, "✅ کد رهگیری `{$tracking_code}` برای سفارش `{$order_id}` ثبت شد و وضعیت به 'ارسال شده' تغییر یافت.", ['remove_keyboard' => true], $platform);

             // Notify user
             $order_owner = getOrderOwner($order_id); // Assumes OrderQueries loaded
             if ($order_owner) {
                 $user_message = "📬 سفارش شما با شماره `{$order_id}` ارسال شد.\nکد رهگیری پستی: `{$tracking_code}`";
                 if (!empty($order_owner['telegram_chat_id'])) {
                     sendMessage($order_owner['telegram_chat_id'], $user_message, null, 'telegram');
                 }
                 if (!empty($order_owner['bale_chat_id'])) {
                     sendMessage($order_owner['bale_chat_id'], $user_message, null, 'bale');
                 }
             }

             // Reset state
             updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'target_order_id' => null]));
             // Optionally show order details again or main menu
             showAdminMainMenu($chat_id, $user_id, $platform);

        } else {
             sendMessage($chat_id, "❌ خطا در ثبت کد رهگیری در دیتابیس. لطفا دوباره تلاش کنید.", null, $platform);
             // Don't reset state on DB error, allow retry
        }
    } else {
         sendMessage($chat_id, "کد رهگیری نمی‌تواند خالی باشد. لطفا مجددا وارد کنید یا انصراف دهید.", null, $platform);
         // Keep state for retry
    }
}

?>
