<?php

if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}

/**
 * Helper function to format weight text for a line item.
 */
function formatLineItemWeight(string $qty, ?string $weight, ?string $unit): string {
    if ($weight !== null && $unit !== null) {
        // محاسبه وزن کل این ردیف (تعداد ضربدر وزن واحد)
        $total_line_weight = bcmul($qty, $weight, 2); // دقت 2 رقم اعشار
        
        // فرمت کردن عدد (حذف اعشار صفر)
        $formatted_weight = number_format((float)$total_line_weight, (strpos($total_line_weight, '.') !== false) ? 2 : 0);
        // حذف .00 اگر وجود داشت برای تمیزی بیشتر
        $formatted_weight = str_replace('.00', '', $formatted_weight);

        return " (وزن کل: {$formatted_weight} {$unit})";
    }
    return "";
}

function convertPersianToEnglishDigits(string $string): string {
    $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    $arabic = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
    $english = range(0, 9);
    $string = str_replace($persian, $english, $string);
    return str_replace($arabic, $english, $string);
}

function translate_status_to_persian(string $status): string {
    $translations = [
        'awaiting_payment' => 'در انتظار پرداخت',
        'awaiting_online_payment' => 'منتظر پرداخت آنلاین',
        'payment_failed' => 'پرداخت ناموفق',
        'processing' => 'در حال پردازش',
        'awaiting_shipping_fee' => 'منتظر هزینه پست',
        'shipped' => 'ارسال شده',
        'completed' => 'تکمیل شده',
        'canceled' => 'لغو شده',
        'refunded' => 'مرجوع شده',
        'payment_rejected' => 'پرداخت رد شد',
        'canceled_card_mismatch' => 'لغو (عدم تطابق کارت)',
        'refund_pending' => 'در انتظار بازگشت وجه',
        'pending' => 'در انتظار',
    ];
    return $translations[$status] ?? ucfirst(str_replace('_', ' ', $status));
}

function buildAdminOrderDetailsText(array $order): string {
    if (empty($order)) return 'خطا: اطلاعات سفارش یافت نشد.';

    $order_id = $order['id'] ?? 'N/A';
    $status_fa = translate_status_to_persian($order['status'] ?? 'نامشخص');
    $date_fa = to_jalali_datetime($order['created_at'] ?? null);
    $user_platform_id = $order['telegram_user_id'] ?? $order['bale_user_id'] ?? 'N/A';
    $user_first_name = htmlspecialchars($order['first_name'] ?? 'نامشخص');
    $user_phone = $order['phone_number'] ?? 'نامشخص';
    $delivery_address = htmlspecialchars($order['delivery_address'] ?? 'ثبت نشده');
    $order_note = isset($order['order_note']) ? htmlspecialchars($order['order_note']) : null;
    $items = $order['items'] ?? [];
    $subtotal = $order['subtotal_amount'] ?? '0.00';
    $discount_amount = $order['discount_amount'] ?? '0.00';
    $discount_code = $order['discount_code'] ?? null;
    $shipping_fee = $order['shipping_fee'] ?? '0.00';
    $shipping_fee_note = isset($order['shipping_fee_note']) ? htmlspecialchars($order['shipping_fee_note']) : null;
    $total_amount_products = $order['total_amount'] ?? '0.00';
    $payment_ref_id = $order['payment_ref_id'] ?? null;
    $payment_card_mask = $order['payment_card_mask'] ?? null;
    $tracking_code = $order['tracking_code'] ?? null;
    $user_internal_id = $order['user_id'] ?? 'N/A';

    $partner_user_id = $order['partner_user_id'] ?? null;
    $partner_name = htmlspecialchars($order['partner_first_name'] ?? '');
    $partner_phone = $order['partner_phone_number'] ?? '';
    $total_partner_profit = $order['total_partner_profit'] ?? '0.00';

    $message = "🔔 *جزئیات سفارش شماره `{$order_id}`*\n\n";
    $message .= "👤 مشتری: {$user_first_name} (ID: `{$user_internal_id}`, Platform ID: `{$user_platform_id}`)\n";
    $message .= "📞 تماس: `{$user_phone}`\n";
    $message .= "🗓 تاریخ: `{$date_fa}`\n\n";

    if ($partner_user_id) {
        $message .= "🤝 *سفارش از طریق همکار:*\n";
        $message .= "  - نام: {$partner_name} (ID: `{$partner_user_id}`)\n";
        $message .= "  - تماس: `{$partner_phone}`\n";
        $message .= "  - 💰 *سود همکار (کل):* `" . number_format($total_partner_profit) . "` تومان\n\n";
    }

    $message .= "📍 *آدرس تحویل:*\n{$delivery_address}\n\n";
    if (!empty($order_note)) {
        $message .= "📝 *یادداشت مشتری:*\n{$order_note}\n\n";
    }
    $message .= "📦 *محصولات سفارش داده شده:*\n";
    if (empty($items)) {
        $message .= "- موردی یافت نشد.\n";
    } else {
        foreach ($items as $item) {
            $qty = $item['quantity'] ?? '0';
            $name = htmlspecialchars($item['product_name'] ?? 'نامشخص');
            $price = number_format($item['price_per_unit'] ?? 0);
            $item_partner_profit = $item['partner_profit'] ?? '0.00';
            
            // اصلاح وزن
            $weight_text = "";
            $weight_val = $item['weight'] ?? null; 
            $weight_unit = $item['weight_unit'] ?? null;

            if ($weight_val) {
                 $weight_text = formatLineItemWeight((string)$qty, (string)$weight_val, (string)$weight_unit);
            }
            
            $message .= "- `{$qty}` واحد از `{$name}` (هر واحد: {$price} تومان){$weight_text}\n";
            if (bccomp($item_partner_profit, '0', BC_SCALE) > 0) {
                $message .= "  (سود همکار: " . number_format($item_partner_profit) . " تومان به ازای هر واحد)\n";
            }
        }
    }

    $message .= "\n💰 *جمع محصولات (قبل تخفیف):* `" . number_format($subtotal) . "` تومان\n";
    if (bccomp($discount_amount, '0', BC_SCALE) > 0) {
        $message .= "📉 *تخفیف" . ($discount_code ? " (`{$discount_code}`)" : '') . ":* `" . number_format($discount_amount) . "` تومان\n";
    }

    $display_total = $total_amount_products;
    if (bccomp($shipping_fee, '0.00', BC_SCALE) > 0) {
        $message .= "🚚 *هزینه پست:* `" . number_format($shipping_fee) . "` تومان\n";
        if (!empty($shipping_fee_note)) {
            $message .= "   (یادداشت: {$shipping_fee_note})\n";
        }
        $display_total = bcadd($total_amount_products, $shipping_fee, BC_SCALE);
    } elseif (bccomp($shipping_fee, '0.00', BC_SCALE) == 0) {
        $message .= "🚚 *هزینه پست:* `رایگان`\n";
        if (!empty($shipping_fee_note)) {
            $message .= "   (یادداشت: {$shipping_fee_note})\n";
        }
    } elseif (bccomp($shipping_fee, '-1.00', BC_SCALE) == 0) {
        $message .= "🚚 *هزینه پست:* `(در انتظار محاسبه)`\n";
    }

    $message .= "💳 *مبلغ نهایی (محصولات + ارسال):* `" . number_format($display_total) . "` تومان\n";

    if (!empty($payment_ref_id)) {
        $message .= "🧾 *شماره پیگیری پرداخت:* `{$payment_ref_id}`\n";
    }
    if (!empty($payment_card_mask)) {
        $message .= "💳 *کارت پرداخت:* `" . $payment_card_mask . "`\n";
    }
    if (!empty($tracking_code)) {
        $message .= "📮 *کد رهگیری پستی:* `{$tracking_code}`\n";
    }
    $message .= "\n🚦 وضعیت فعلی: *{$status_fa}*";

    return $message;
}

function buildCustomerOrderDetailsText(array $order): string {
    if (empty($order)) return 'خطا: اطلاعات سفارش یافت نشد.';

    $order_id = $order['id'] ?? 'N/A';
    $status_fa = translate_status_to_persian($order['status'] ?? 'نامشخص');
    $date_fa = to_jalali_datetime($order['created_at'] ?? null);
    $delivery_address = htmlspecialchars($order['delivery_address'] ?? 'ثبت نشده');
    $items = $order['items'] ?? [];
    $subtotal = $order['subtotal_amount'] ?? '0.00';
    $discount_amount = $order['discount_amount'] ?? '0.00';
    $discount_code = $order['discount_code'] ?? null;
    $shipping_fee = $order['shipping_fee'] ?? '0.00';
    $shipping_fee_note = isset($order['shipping_fee_note']) ? htmlspecialchars($order['shipping_fee_note']) : null;
    $total_amount_products = $order['total_amount'] ?? '0.00';
    $payment_ref_id = $order['payment_ref_id'] ?? null;
    $tracking_code = $order['tracking_code'] ?? null;

    $message = "📄 *جزئیات سفارش شما به شماره `{$order_id}`*\n\n";
    $message .= "🗓 *تاریخ ثبت:* `{$date_fa}`\n";
    $message .= "🚦 *وضعیت فعلی:* *{$status_fa}*\n\n";
    if (!empty($tracking_code)) {
        $message .= "📮 *کد رهگیری پستی:* `{$tracking_code}`\n\n";
    }
    if (!empty($payment_ref_id)) {
        $message .= "🧾 *شماره پیگیری پرداخت (کالا):* `{$payment_ref_id}`\n\n";
    }
    $message .= "📦 *محصولات سفارش داده شده:*\n";
    if (empty($items)) {
        $message .= "- موردی یافت نشد.\n";
    } else {
        foreach ($items as $item) {
            $qty = $item['quantity'] ?? '0';
            $name = htmlspecialchars($item['product_name'] ?? 'نامشخص');
            $item_total = bcmul((string)($item['quantity'] ?? 0), (string)($item['price_per_unit'] ?? 0), BC_SCALE);
            
            $weight_text = "";
            if (isset($item['weight']) && isset($item['weight_unit'])) {
                 $weight_text = formatLineItemWeight((string)$qty, (string)$item['weight'], (string)$item['weight_unit']);
            }

            $message .= "- `{$qty}` واحد از *{$name}*{$weight_text}\n   قیمت کل: `" . number_format($item_total) . "` تومان\n";
        }
    }

    $message .= "-------------------------------------\n";
    $message .= "💰 *جمع محصولات:* `" . number_format($subtotal) . "` تومان\n";
    if (bccomp($discount_amount, '0', BC_SCALE) > 0) {
        $message .= "📉 *تخفیف" . ($discount_code ? " (`{$discount_code}`)" : '') . ":* `" . number_format($discount_amount) . "` تومان\n";
    }

    $display_total = $total_amount_products;
    if (bccomp($shipping_fee, '0.00', BC_SCALE) > 0) {
        $message .= "🚚 *هزینه پست:* `" . number_format($shipping_fee) . "` تومان\n";
        if (!empty($shipping_fee_note)) {
            $message .= "   (یادداشت: {$shipping_fee_note})\n";
        }
        $display_total = bcadd($total_amount_products, $shipping_fee, BC_SCALE);
    } elseif (bccomp($shipping_fee, '0.00', BC_SCALE) == 0) {
        $message .= "🚚 *هزینه پست:* `رایگان`\n";
        if (!empty($shipping_fee_note)) {
            $message .= "   (یادداشت: {$shipping_fee_note})\n";
        }
    } elseif (bccomp($shipping_fee, '-1.00', BC_SCALE) == 0) {
        $message .= "🚚 *هزینه پست:* `(در انتظار محاسبه)`\n";
    }

    $message .= "💳 *مبلغ نهایی (محصولات + ارسال):* `" . number_format($display_total) . "` تومان\n\n";
    $message .= "📍 *آدرس تحویل:*\n`{$delivery_address}`\n";

    if (in_array($order['status'] ?? '', ['awaiting_online_payment', 'payment_failed'])) {
        $message .= "\n⚠️ *پرداخت کالا ناموفق بود یا کامل نشده است.*";
        if (!empty($order['payment_authority']) || $order['status'] === 'awaiting_online_payment') {
            $message .= " برای تلاش مجدد، از طریق دکمه زیر اقدام کنید.";
        } else {
            $message .= " لطفا با پشتیبانی تماس بگیرید.";
        }
    }

    return $message;
}

function buildPartnerCustomerOrderDetailsText(array $order): string {
    if (empty($order)) return 'خطا: اطلاعات سفارش یافت نشد.';

    $order_id = $order['id'] ?? 'N/A';
    $status_fa = translate_status_to_persian($order['status'] ?? 'نامشخص');
    $date_fa = to_jalali_datetime($order['created_at'] ?? null);
    $user_first_name = htmlspecialchars($order['first_name'] ?? 'نامشخص');
    $items = $order['items'] ?? [];
    $total_amount_products = $order['total_amount'] ?? '0.00';
    $shipping_fee = $order['shipping_fee'] ?? '0.00';
    $total_partner_profit = $order['total_partner_profit'] ?? '0.00';
    $is_profit_paid = $order['is_profit_paid'] ?? false;
    $payout_request_id = $order['payout_request_id'] ?? null;

    $message = "📈 *جزئیات سفارش مشتری شما (شماره `{$order_id}`)*\n\n";
    $message .= "👤 مشتری: *{$user_first_name}*\n";
    $message .= "🗓 تاریخ: `{$date_fa}`\n";
    $message .= "🚦 وضعیت سفارش: *{$status_fa}*\n\n";

    $message .= "📦 *محصولات:*\n";
    if (empty($items)) {
        $message .= "- موردی یافت نشد.\n";
    } else {
        foreach ($items as $item) {
            $qty = $item['quantity'] ?? '?';
            $name = htmlspecialchars($item['product_name'] ?? 'نامشخص');
            $item_partner_profit = $item['partner_profit'] ?? '0.00';
            $item_total_profit = bcmul($qty, $item_partner_profit, BC_SCALE);
            
            $message .= "- `{$qty}` واحد *{$name}*\n  (سود شما: `{$qty}` × " . number_format($item_partner_profit) . " = *" . number_format($item_total_profit) . "* تومان)\n";
        }
    }

    $message .= "\n-------------------------------------\n";
    $message .= "💰 *جمع کل سود شما از این سفارش:* `" . number_format($total_partner_profit) . "` تومان\n";

    $display_total = $total_amount_products;
    if (bccomp($shipping_fee, '0.00', BC_SCALE) > 0) {
        $display_total = bcadd($total_amount_products, $shipping_fee, BC_SCALE);
    }

    $message .= "💳 *مبلغ کل پرداختی مشتری:* `" . number_format($display_total) . "` تومان\n\n";

    if ($is_profit_paid && $payout_request_id) {
        $message .= "✅ *وضعیت سود: تسویه شده* (درخواست `{$payout_request_id}`)";
    } elseif ($payout_request_id) {
        $message .= "⏳ *وضعیت سود: در انتظار تسویه* (درخواست `{$payout_request_id}`)";
    } elseif (in_array($order['status'], ['completed', 'shipped', 'processing'])) {
        $message .= "🟢 *وضعیت سود: قابل برداشت*";
    } else {
        $message .= "🟡 *وضعیت سود: در انتظار تکمیل سفارش* (سود پس از تکمیل سفارش قابل برداشت خواهد بود)";
    }

    return $message;
}

function buildOrderSummaryTextInternal(int $user_id, array $session_data, array $user): string {
    $checkout_data = $session_data['checkout'] ?? [];
    $cart_items = getCartContents($user_id);
    $subtotal = '0.00';

    $summary = "📋 *پیش‌فاکتور سفارش شما*\n\n";
    if (empty($cart_items)) return "سبد خرید شما خالی است.";

    foreach ($cart_items as $item) {
        $qty = (string)($item['quantity'] ?? '0');
        $price = (string)($item['pricing']['price'] ?? '0.00');
        $unit = $item['unit'] ?? '';
        $name = htmlspecialchars($item['name'] ?? 'نامشخص');
        
        $item_weight = $item['weight'] ?? null;
        $item_weight_unit = $item['weight_unit'] ?? null;
        
        $item_subtotal = bcmul($qty, $price, BC_SCALE);
        $subtotal = bcadd($subtotal, $item_subtotal, BC_SCALE);
        
        // اصلاح وزن
        $weight_text = formatLineItemWeight($qty, $item_weight, $item_weight_unit);

        $summary .= "🔹 *{$name}*: `" . number_format($item_subtotal) . "` تومان (`{$qty}` {$unit}){$weight_text}\n";
    }

    $summary .= "-------------------------------------\n";
    $summary .= "*جمع محصولات:* `" . number_format($subtotal) . "` تومان\n";
    
    $discount_amount = $checkout_data['discount_amount'] ?? '0.00';
    if (bccomp($discount_amount, '0', BC_SCALE) > 0) {
        $discount_code = $checkout_data['discount_code'] ?? '';
        $summary .= "📉 *تخفیف" . ($discount_code ? " (`{$discount_code}`)" : '') . ":* `-" . number_format($discount_amount) . "` تومان\n";
    }

    $summary .= "🚚 *هزینه ارسال:* `(پس از بررسی توسط مدیریت محاسبه می‌شود)`\n";

    $payable_amount_before_shipping = bcsub($subtotal, $discount_amount, BC_SCALE);
    $total_payable = $payable_amount_before_shipping; 
    if (bccomp($total_payable, '0', BC_SCALE) < 0) { $total_payable = '0.00'; }
    
    $summary .= "💳 *مبلغ نهایی قابل پرداخت (فقط کالا):* `" . number_format($total_payable) . "` تومان\n\n";
    
    $is_partner_customer = isset($user['referred_by_user_id']) && $user['referred_by_user_id'] !== null;
    $selected_card_id = $session_data['selected_card_id'] ?? null;

    if ($is_partner_customer) {
        $summary .= "💳 *کارت پرداخت:* (پرداخت با هر کارتی مجاز است)\n";
    } elseif ($selected_card_id) {
         $cards = getUserCards($user_id);
         $selected_card_mask = '';
         foreach ($cards as $card) { if (($card['id'] ?? 0) == $selected_card_id) { $selected_card_mask = $card['card_number_masked'] ?? '****'; break; } }
         $summary .= "💳 *کارت پرداخت:* `" . ($selected_card_mask ?: '*خطا*') . "`\n";
    } else {
        $summary .= "💳 *کارت پرداخت:* *انتخاب نشده است*\n";
    }

    if(!empty($checkout_data['province']) && !empty($checkout_data['city']) && !empty($checkout_data['address_text']) && !empty($checkout_data['postal_code'])) {
       $summary .= "📍 *آدرس:* {$checkout_data['province']}, {$checkout_data['city']}, " . htmlspecialchars($checkout_data['address_text']) . " (کدپستی: {$checkout_data['postal_code']})\n";
    } else {
         $summary .= "📍 *آدرس:* *هنوز وارد نشده است*\n";
    }

    if(!empty($checkout_data['note'])) {
        $summary .= "📝 *یادداشت:* " . htmlspecialchars($checkout_data['note']) . "\n";
    }

    return $summary;
}

function buildPaymentConfirmationButtonsInternal(int $user_id, array $session_data, array $user): array {
    $checkout_data = $session_data['checkout'] ?? [];
    $buttons = [];
    $address_complete = !empty($checkout_data['province']) && !empty($checkout_data['city']) && !empty($checkout_data['address_text']) && !empty($checkout_data['postal_code']);
    
    $is_partner_customer = isset($user['referred_by_user_id']) && $user['referred_by_user_id'] !== null;
    $selected_card_id = $session_data['selected_card_id'] ?? null;
    $card_selected = ($is_partner_customer && $selected_card_id === null) || (!$is_partner_customer && !empty($selected_card_id));
    
    $cart_exists = !empty(getCartContents($user_id));

    if($address_complete && $card_selected && $cart_exists) {
        $buttons[] = [['text' => '✅ تایید نهایی و پرداخت', 'callback_data' => buildCallbackData('confirmOrderBeforePayment')]];
    } else {
         $missing = [];
         if (!$cart_exists) $missing[] = 'سبد خالی';
         if (!$address_complete) $missing[] = 'آدرس ناقص';
         if (!$card_selected) $missing[] = 'کارت انتخاب نشده';
       $buttons[] = [['text' => '❌ اطلاعات ناقص: ' . implode('، ', $missing), 'callback_data' => 'noop']];
    }

    if (!$is_partner_customer) {
        if (isset($checkout_data['discount_code'])) {
            $buttons[] = [['text' => '❌ حذف تخفیف', 'callback_data' => buildCallbackData('removeDiscount')]];
        } else {
            $buttons[] = [['text' => '🎟 ثبت کد تخفیف', 'callback_data' => buildCallbackData('applyDiscount')]];
        }
    }
    
    $buttons[] = [['text' => '✏️ ویرایش آدرس/یادداشت', 'callback_data' => buildCallbackData('backToProvinceSelection')]];
    
    if (!$is_partner_customer) {
        $buttons[] = [['text' => '💳 تغییر کارت پرداخت', 'callback_data' => buildCallbackData('checkoutStart')]];
    }
    
    $buttons[] = [['text' => '🛒 بازگشت به سبد خرید', 'callback_data' => buildCallbackData('viewCart')]];

    return $buttons;
}

function buildPayoutRequestAdminText(array $request): string {
    if (empty($request)) return 'خطا: اطلاعات درخواست یافت نشد.';

    $request_id = $request['id'] ?? 'N/A';
    $partner_name = htmlspecialchars($request['first_name'] ?? 'نامشخص');
    $partner_phone = $request['phone_number'] ?? 'نامشخص';
    $amount = $request['amount'] ?? '0.00';
    $card_masked = $request['card_number_masked'] ?? 'نامشخص';
    $date_fa = to_jalali_datetime($request['created_at'] ?? null);

    $message = "💳 *درخواست تسویه حساب جدید* (شماره `{$request_id}`)\n\n";
    $message .= "👤 همکار: *{$partner_name}*\n";
    $message .= "📞 تماس: `{$partner_phone}`\n";
    $message .= "🗓 تاریخ درخواست: `{$date_fa}`\n\n";
    $message .= "💰 *مبلغ درخواست:* `" . number_format($amount) . "` تومان\n";
    $message .= "🏦 *کارت مقصد:*\n`{$card_masked}`\n\n";
    $message .= "لطفا پس از بررسی و واریز وجه، دکمه «تایید پرداخت» را بزنید.";

    return $message;
}
?>